// PIC32MZ_11.c - Using the Helix library to play MP3s
// by Aidan Mocke 2018-11-20
//
// Credit:  Naturally, I didn't write the Helix MP3 decoder It can be found at https://www.helixcommunity.org/projects/datatype/mp3dec
//          I made extensive use of the information provided by Microchip at https://www.microchip.com/SWLibraryWeb/product.aspx?product=Helix%20MP3%20Decoder%20Library
//          The Helix MP3 code is based heavily on code I found at https://github.com/derkling/adplayer/tree/master/HELIX

// PIC32MZ2048EFH144 Configuration Bit Settings

// 'C' source line config statements

// DEVCFG3
// USERID = No Setting
#pragma config FMIIEN = ON              // Ethernet RMII/MII Enable (MII Enabled)
#pragma config FETHIO = ON              // Ethernet I/O Pin Select (Default Ethernet I/O)
#pragma config PGL1WAY = OFF            // Permission Group Lock One Way Configuration (Allow multiple reconfigurations)
#pragma config PMDL1WAY = OFF           // Peripheral Module Disable Configuration (Allow multiple reconfigurations)
#pragma config IOL1WAY = OFF            // Peripheral Pin Select Configuration (Allow multiple reconfigurations)
#pragma config FUSBIDIO = OFF           // USB USBID Selection (Controlled by Port Function)

// DEVCFG2
#pragma config FPLLIDIV = DIV_1         // System PLL Input Divider (3x Divider)
#pragma config FPLLRNG = RANGE_5_10_MHZ // System PLL Input Range (5-10 MHz Input)
#pragma config FPLLICLK = PLL_FRC      // System PLL Input Clock Selection (POSC is input to the System PLL)
#pragma config FPLLMULT = MUL_50        // System PLL Multiplier (PLL Multiply by 50)
#pragma config FPLLODIV = DIV_2         // System PLL Output Clock Divider (2x Divider)
#pragma config UPLLFSEL = FREQ_24MHZ    // USB PLL Input Frequency Selection (USB PLL input is 24 MHz)

// DEVCFG1
#pragma config FNOSC = SPLL             // Oscillator Selection Bits (System PLL)
#pragma config DMTINTV = WIN_127_128    // DMT Count Window Interval (Window/Interval value is 127/128 counter value)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disable SOSC)
#pragma config IESO = OFF               // Internal/External Switch Over (Disabled)
#pragma config POSCMOD = OFF             // Primary Oscillator Configuration (External clock mode)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FCKSM = CSECME           // Clock Switching and Monitor Selection (Clock Switch Enabled, FSCM Enabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WDTSPGM = STOP           // Watchdog Timer Stop During Flash Programming (WDT stops during Flash programming)
#pragma config WINDIS = NORMAL          // Watchdog Timer Window Mode (Watchdog Timer is in non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled)
#pragma config FWDTWINSZ = WINSZ_25     // Watchdog Timer Window Size (Window size is 25%)
#pragma config DMTCNT = DMT31           // Deadman Timer Count Selection (2^31 (2147483648))
#pragma config FDMTEN = OFF             // Deadman Timer Enable (Deadman Timer is disabled)

// DEVCFG0
#pragma config DEBUG = OFF              // Background Debugger Enable (Debugger is disabled)
#pragma config JTAGEN = OFF             // JTAG Enable (JTAG Disabled)
#pragma config ICESEL = ICS_PGx1        // ICE/ICD Comm Channel Select (Communicate on PGEC1/PGED1)
#pragma config TRCEN = OFF              // Trace Enable (Trace features in the CPU are disabled)
#pragma config BOOTISA = MIPS32         // Boot ISA Selection (Boot code and Exception code is MIPS32)
#pragma config FECCCON = OFF_UNLOCKED   // Dynamic Flash ECC Configuration (ECC and Dynamic ECC are disabled (ECCCON bits are writable))
#pragma config FSLEEP = OFF             // Flash Sleep Mode (Flash is powered down when the device is in Sleep mode)
#pragma config DBGPER = PG_ALL          // Debug Mode CPU Access Permission (Allow CPU access to all permission regions)
#pragma config SMCLR = MCLR_NORM        // Soft Master Clear Enable bit (MCLR pin generates a normal system Reset)
#pragma config SOSCGAIN = GAIN_2X       // Secondary Oscillator Gain Control bits (2x gain setting)
#pragma config SOSCBOOST = ON           // Secondary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config POSCGAIN = GAIN_2X       // Primary Oscillator Gain Control bits (2x gain setting)
#pragma config POSCBOOST = ON           // Primary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config EJTAGBEN = NORMAL        // EJTAG Boot (Normal EJTAG functionality)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)


#include <xc.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include "ff.h"
#include "diskio.h"
#include <string.h>
#include "coder.h"
#include "mp3dec.h"


#define SYS_FREQ 200000000

#define PLAYBACK_BUFFER_SIZE 2304 * 2                  // Output buffer is 4,608 32-bit words, four times the MP3 decode buffer size

#define MP3_BUF_SIZE 1940

unsigned char MP3_READ_BUFFER[MP3_BUF_SIZE];            // Stores a chunk read from the MP3 file
unsigned char *MP3_READ_BUFFER_PTR;                     // Stores where we are in MP3_READ_BUFFER
signed short MP3_OUTPUT_BUFFER[1152 * 2];               // Stores the output from the MP3 decoder

int playback_buffer[PLAYBACK_BUFFER_SIZE];              // The actual data to play back
volatile int pb_readpos;                                // The read position in the playback buffer
volatile int pb_writepos;                               // The write position in the playback buffer

HMP3Decoder mp3Decoder;                                 // The MP3 decoder itself

FIL file;                                               // file is the handle of the MP3 file we are streaming from SD card
DIR dir;                                                // Stores information about the current directory
FATFS fso;                                              // Stores information about the file system
FILINFO fileInfo;                                       // Stores information about the current file

long set_performance_mode()
{   
	unsigned int cp0;
	
    // Unlock Sequence
    asm volatile("di"); // Disable all interrupts
    SYSKEY = 0xAA996655;
    SYSKEY = 0x556699AA;  

    // PB1DIV
    // Peripheral Bus 1 cannot be turned off, so there's no need to turn it on
    PB1DIVbits.PBDIV = 1; // Peripheral Bus 1 Clock Divisor Control (PBCLK1 is SYSCLK divided by 2)

    // PB2DIV
    PB2DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB2DIVbits.PBDIV = 1; // Peripheral Bus 2 Clock Divisor Control (PBCLK2 is SYSCLK divided by 2)

    // PB3DIV
    PB3DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB3DIVbits.PBDIV = 1; // Peripheral Bus 3 Clock Divisor Control (PBCLK3 is SYSCLK divided by 2)

    // PB4DIV
    PB4DIVbits.ON = 1; // Peripheral Bus 4 Output Clock Enable (Output clock is enabled)
    while (!PB4DIVbits.PBDIVRDY); // Wait until it is ready to write to
    PB4DIVbits.PBDIV = 0; // Peripheral Bus 4 Clock Divisor Control (PBCLK4 is SYSCLK divided by 1)

    // PB5DIV
    PB5DIVbits.ON = 1; // Peripheral Bus 5 Output Clock Enable (Output clock is enabled)
    PB5DIVbits.PBDIV = 1; // Peripheral Bus 5 Clock Divisor Control (PBCLK5 is SYSCLK divided by 2)

    // PB7DIV
    PB7DIVbits.ON = 1; // Peripheral Bus 7 Output Clock Enable (Output clock is enabled)
    PB7DIVbits.PBDIV = 0; // Peripheral Bus 7 Clock Divisor Control (PBCLK7 is SYSCLK divided by 1)

    // PB8DIV
    PB8DIVbits.ON = 1; // Peripheral Bus 8 Output Clock Enable (Output clock is enabled)
    PB8DIVbits.PBDIV = 1; // Peripheral Bus 8 Clock Divisor Control (PBCLK8 is SYSCLK divided by 2)

    // PRECON - Set up prefetch
    PRECONbits.PFMSECEN = 0; // Flash SEC Interrupt Enable (Do not generate an interrupt when the PFMSEC bit is set)
    PRECONbits.PREFEN = 0b11; // Predictive Prefetch Enable (Enable predictive prefetch for any address)
    PRECONbits.PFMWS = 0b010; // PFM Access Time Defined in Terms of SYSCLK Wait States (Two wait states)

    // Set up caching
    cp0 = _mfc0(16, 0);
    cp0 &= ~0x07;
    cp0 |= 0b011; // K0 = Cacheable, non-coherent, write-back, write allocate
    _mtc0(16, 0, cp0);  

    // Lock Sequence
    SYSKEY = 0x33333333;
    
    asm volatile("ei"); // Enable all interrupts
}

void setup_ports()
{
    // Turn off all analog inputs, make everything digital
    ANSELA = 0;
    ANSELB = 0;
    ANSELC = 0;
    ANSELD = 0;
    ANSELE = 0;
    ANSELF = 0;
    ANSELG = 0;
    ANSELH = 0;
    ANSELJ = 0;
    
    // Make all ports outputs initially
    TRISA = 0;
    TRISB = 0;
    TRISC = 0;
    TRISD = 0;
    TRISE = 0;
    TRISF = 0;
    TRISG = 0;
    TRISH = 0;
    TRISJ = 0;        
}

void initDisk()
{// Wait for the disk to initialise
    while(disk_initialize(0));
    // Mount the disk
    f_mount(&fso, "/", 0);
    // Change dir to the root directory
    f_chdir("/");
    // Open the directory
    f_opendir(&dir, ".");
}

int find_first_frame()
{
    int nRead;
    
    int offset;
    
    nRead = 1;
    
    // Find a valid MP3 start of frame in the input stream
    while(nRead > 0)
    {
        // Read the input file
        f_read(&file, MP3_READ_BUFFER, MP3_BUF_SIZE, &nRead);
        if(nRead == 0)
        {
            // We have reached end of file and a valid MP3 start of frame was not found.
            return -1;
        }
        else
        {
            offset = MP3FindSyncWord(MP3_READ_BUFFER, MP3_BUF_SIZE);
            if(offset < 0)
            {
                // The input buffer does not contain a start of frame. Read another frame.
                continue;
            }
            else
            {
                // We found a start of frame. offset contains location of the start of frame
                 // within input buffer.
                return offset;
            }
        }
    }    
}

static int fill_read_buffer(unsigned char *readBuf, unsigned char *readPtr, int bufSize, int bytesLeft, FIL *infile)
{
    /* This function will move bytesLeft bytes from
     * readPtr to the top of readBuf. bufSize is the
     * size of readBuf. It then reads bufSize - bytesLeft bytes
     * from infile and appends these bytes to readBuf. 
     * If readBuf is not full, then remaing bytes are
     * set to zero. The total number of bytes read from 
     * infile is returned.
     */

    int nRead;

    /* move last, small chunk from end of buffer to start, then fill with new data */
    memmove(readBuf, readPtr, bytesLeft);				
    
    f_read(&file, readBuf + bytesLeft, bufSize - bytesLeft, &nRead);

    /* zero-pad to avoid finding false sync word after last frame (from old data in readBuf) */
    if (nRead < bufSize - bytesLeft)
        memset(readBuf + bytesLeft + nRead, 0, bufSize - bytesLeft - nRead);	

    return nRead;
}

void __attribute__((vector(_TIMER_3_VECTOR), interrupt(ipl5soft), nomips16)) timer3_handler()
{
	IFS0bits.T3IF = 0;	// Clear interrupt flag for timer 3

    // Play out the left channel first
    OC2RS = playback_buffer[pb_readpos];
    
    // Increment the read position in the playback buffer
    pb_readpos++;
    
    // Check if the read position has exceeded the length of the playback buffer. If so, restart it at 0.
     if (pb_readpos >= PLAYBACK_BUFFER_SIZE) 
         pb_readpos = 0;

    // Now play out the right channel
    OC6RS = playback_buffer[pb_readpos];

    // Increment the read position in the playback buffer
    pb_readpos++;

    // Check if the read position has exceeded the length of the playback buffer. If so, restart it at 0.
     if (pb_readpos >= PLAYBACK_BUFFER_SIZE) 
         pb_readpos = 0;
}

void PWM_init(int frequency)
{
    T3CON	= 0x0;		// Disable timer 3 when setting it up
    TMR3	= 0;		// Set timer 3 counter to 0
    IEC0bits.T3IE = 0;	// Disable Timer 3 Interrupt
    
    // Set up the period. Period = PBCLK3 frequency, which is SYS_FREQ / 2, divided by the frequency we want to output the audio at
    PR3 = SYS_FREQ / 2 / frequency;
		
	IFS0bits.T3IF = 0;	// Clear interrupt flag for timer 3
    IPC3bits.T3IP = 5; 	// Interrupt priority 5
    IPC3bits.T3IS = 1; 	// Sub-priority 1
    IEC0bits.T3IE = 1;	// Enable Timer 3 Interrupt


	OC2CON = 0;				// Turn off Output Compare module 6
    OC2CONbits.OCTSEL = 1; 	// Interrupt source for this module is Timer 3
    OC2CONbits.OCM = 0b110;	// Output Compare Mode (OCM) is 6, which is PWM mode
    OC2RS = 0;				// Keep the signal low for the entire duration

	OC6CON = 0;				// Turn off Output Compare module 6
    OC6CONbits.OCTSEL = 1; 	// Interrupt source for this module is Timer 3
    OC6CONbits.OCM = 0b110;	// Output Compare Mode (OCM) is 6, which is PWM mode
    OC6RS = 0;				// Keep the signal low for the entire duration
    OC2CONbits.ON = 1;		// Turn on Output Compare module 6 (OC6)
    OC6CONbits.ON = 1;		// Turn on Output Compare module 6 (OC6)    

	// Turn on timer 3
	T3CONbits.TON	= 1;
}

void PWM_stop()
{
    // Stop Timer 3 and the PWM outputs
    T3CONbits.TON = 0;
    OC2CONbits.ON = 0;
    OC6CONbits.ON = 0;
}

int main()
{
    int count;
    int error;
    int dec_error;
    MP3FrameInfo frameInfo;
    int bytesLeft;
    int offset;
    int nRead;
    char F_EOF;
    int convert;
    
    set_performance_mode();
    
    setup_ports();
        
    // PPS for SD card
    RPB5R = 0b0110; 		// Set SDO2 = RB5
    TRISBbits.TRISB3 = 1; 	// Set SDI2 pin to input
    SDI2R = 0b1000; 		// SDI2 = RB3    
    
    // Enabling internal pull-up bits to counter potential problems
    CNPUBbits.CNPUB3 = 1;
    CNPUBbits.CNPUB5 = 1;
    
	// PPS for PWM outputs
    RPC1R = 0b1100;			// RC1 = OC6
    RPC2R = 0b1011;			// RC2 = OC2
	
    // Enable multi-vectored interrupts mode
    INTCONbits.MVEC = 1;
    
    // Initialise SD card
    initDisk();
    
    // Reset SPI2BRG to give us maximum speed on SPI2 (50MHz)
    SPI2BRG = 0;
    
    // Open our MP3 file
    f_open(&file, "/beat.mp3", FA_READ);

    // Initialise the MP3 decoder
    mp3Decoder = MP3InitDecoder();
        
    if(mp3Decoder == 0)
    {
    // This means the memory allocation failed. This typically happens if there is not
    // enough heap memory. Recompile the code with additional heap memory.
    }
    
    // Not at the end of the file yet
    F_EOF = 0;    
    
    // Playback buffer read position is 0
    pb_readpos = 0;
    
    // Playback buffer write position is 1
    pb_writepos = 1;

    // Loop until we find a valid MP3 frame
    do
    {
        offset = -1;

        // Loop until we find a sync word
        while (offset < 0)
        {
            // Read a chunk from the file
            f_read(&file, MP3_READ_BUFFER, MP3_BUF_SIZE, &bytesLeft);
            MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;

            // If we've read 0 bytes, this means we've reached the end of the file or some error occurred
            if(bytesLeft == 0)
            {
                F_EOF = 1;
                break;
            }

            // Attempt to find the sync word in the MP3 file
            offset = MP3FindSyncWord(MP3_READ_BUFFER_PTR, MP3_BUF_SIZE);

            if(offset < 0)
            {
                // We did not find an offset, point MP3 read buffer pointer to the start of the array
                MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
            }
            else
            {
                // We found an offset, point MP3 read buffer to the start offset of the sync word
                MP3_READ_BUFFER_PTR += offset;
                // We can discard the data before the sync word
                bytesLeft -= offset;
                // Get out of the while() loop
                break;
            }
        }

        offset = -1;

        // Attempt to get frame information from the read buffer
        error = MP3GetNextFrameInfo(mp3Decoder, &frameInfo, MP3_READ_BUFFER_PTR);
    }
    while (error != 0);

    // Once we get to this point we've found a valid MP3 frame

    // Initialise Timer 3 and PWM to use this new frequency
    PWM_init(frameInfo.samprate);

    // Fill up the MP3 read buffer if necessary
    if (bytesLeft < MP3_BUF_SIZE)
    {
        // Fill the MP3 read buffer
        nRead = fill_read_buffer(MP3_READ_BUFFER, MP3_READ_BUFFER_PTR, MP3_BUF_SIZE, bytesLeft, &file);
        // If no data was read, we've reached the end of the file
        if (nRead == 0) F_EOF = 1;
        // Add numbers of bytes read to the total bytes left in the read buffer
        bytesLeft += nRead;
        // Point the read buffer pointer to the beginning of the array again
        MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
    }

    // Once we get to this point the actual decoding of the MP3 begins. 
    // From here it should just be about reading and decoding the MP3 and then putting that data into the playback buffer
    dec_error = MP3Decode(mp3Decoder, &MP3_READ_BUFFER_PTR, &bytesLeft, MP3_OUTPUT_BUFFER, 0);

    while ((dec_error == 0) && (!F_EOF))
    {                    
        // We have a decoded buffer, let's put it into the playback buffer
        for (count = 0; count < 2304; count++)
        {
            // Do not overwrite data that has not been read yet
            while (pb_writepos == pb_readpos);

            // MP3 data is 16-bit signed but we need to output 11-bit unsigned so convert it
            playback_buffer[pb_writepos] = ((MP3_OUTPUT_BUFFER[count] + 32768) >> 5);
            
            // Increment the position in the playback buffer
            pb_writepos++;

            // If the write position is out of bounds, reset it back to the beginning
            if (pb_writepos >= PLAYBACK_BUFFER_SIZE) 
            {
                pb_writepos = 0;
            }
        }

        // Do we have space left in the playback buffer?
        if (bytesLeft < MP3_BUF_SIZE)
        {
            // Fill the MP3 read buffer
            nRead = fill_read_buffer(MP3_READ_BUFFER, MP3_READ_BUFFER_PTR, MP3_BUF_SIZE, bytesLeft, &file);
            // If no data was read, we've reached the end of the file
            if (nRead == 0) F_EOF = 1;
            // Add numbers of bytes read to the total bytes left in the read buffer
            bytesLeft += nRead;
            // Point the read buffer pointer to the beginning of the array again
            MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
        }            

        // Attempt to decode the data in the MP3 read buffer
        dec_error = MP3Decode(mp3Decoder, &MP3_READ_BUFFER_PTR, &bytesLeft, MP3_OUTPUT_BUFFER, 0);
    }
    
    // Stop the PWM outputs
    PWM_stop();
    
    // Close the file
    f_close(&file);
                
    // Release the decoder from memorry
    MP3FreeDecoder(mp3Decoder);
}