// PIC32MZ_12.c - 
// by Aidan Mocke 2018-11-22
//
// Credit:  Naturally, I didn't write the Helix MP3 decoder It can be found at https://www.helixcommunity.org/projects/datatype/mp3dec
//          I made extensive use of the information provided by Microchip at https://www.microchip.com/SWLibraryWeb/product.aspx?product=Helix%20MP3%20Decoder%20Library
//          The Helix MP3 code is based heavily on code I found at https://github.com/derkling/adplayer/tree/master/HELIX

// PIC32MZ2048EFH144 Configuration Bit Settings

// 'C' source line config statements

// DEVCFG3
// USERID = No Setting
#pragma config FMIIEN = ON              // Ethernet RMII/MII Enable (MII Enabled)
#pragma config FETHIO = ON              // Ethernet I/O Pin Select (Default Ethernet I/O)
#pragma config PGL1WAY = OFF            // Permission Group Lock One Way Configuration (Allow multiple reconfigurations)
#pragma config PMDL1WAY = OFF           // Peripheral Module Disable Configuration (Allow multiple reconfigurations)
#pragma config IOL1WAY = OFF            // Peripheral Pin Select Configuration (Allow multiple reconfigurations)
#pragma config FUSBIDIO = OFF           // USB USBID Selection (Controlled by Port Function)

// DEVCFG2
#pragma config FPLLIDIV = DIV_1         // System PLL Input Divider (3x Divider)
#pragma config FPLLRNG = RANGE_5_10_MHZ // System PLL Input Range (5-10 MHz Input)
#pragma config FPLLICLK = PLL_FRC      // System PLL Input Clock Selection (POSC is input to the System PLL)
#pragma config FPLLMULT = MUL_50        // System PLL Multiplier (PLL Multiply by 50)
#pragma config FPLLODIV = DIV_2         // System PLL Output Clock Divider (2x Divider)
#pragma config UPLLFSEL = FREQ_24MHZ    // USB PLL Input Frequency Selection (USB PLL input is 24 MHz)

// DEVCFG1
#pragma config FNOSC = SPLL             // Oscillator Selection Bits (System PLL)
#pragma config DMTINTV = WIN_127_128    // DMT Count Window Interval (Window/Interval value is 127/128 counter value)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disable SOSC)
#pragma config IESO = OFF               // Internal/External Switch Over (Disabled)
#pragma config POSCMOD = OFF             // Primary Oscillator Configuration (External clock mode)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FCKSM = CSECME           // Clock Switching and Monitor Selection (Clock Switch Enabled, FSCM Enabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WDTSPGM = STOP           // Watchdog Timer Stop During Flash Programming (WDT stops during Flash programming)
#pragma config WINDIS = NORMAL          // Watchdog Timer Window Mode (Watchdog Timer is in non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled)
#pragma config FWDTWINSZ = WINSZ_25     // Watchdog Timer Window Size (Window size is 25%)
#pragma config DMTCNT = DMT31           // Deadman Timer Count Selection (2^31 (2147483648))
#pragma config FDMTEN = OFF             // Deadman Timer Enable (Deadman Timer is disabled)

// DEVCFG0
#pragma config DEBUG = OFF              // Background Debugger Enable (Debugger is disabled)
#pragma config JTAGEN = OFF             // JTAG Enable (JTAG Disabled)
#pragma config ICESEL = ICS_PGx1        // ICE/ICD Comm Channel Select (Communicate on PGEC1/PGED1)
#pragma config TRCEN = OFF              // Trace Enable (Trace features in the CPU are disabled)
#pragma config BOOTISA = MIPS32         // Boot ISA Selection (Boot code and Exception code is MIPS32)
#pragma config FECCCON = OFF_UNLOCKED   // Dynamic Flash ECC Configuration (ECC and Dynamic ECC are disabled (ECCCON bits are writable))
#pragma config FSLEEP = OFF             // Flash Sleep Mode (Flash is powered down when the device is in Sleep mode)
#pragma config DBGPER = PG_ALL          // Debug Mode CPU Access Permission (Allow CPU access to all permission regions)
#pragma config SMCLR = MCLR_NORM        // Soft Master Clear Enable bit (MCLR pin generates a normal system Reset)
#pragma config SOSCGAIN = GAIN_2X       // Secondary Oscillator Gain Control bits (2x gain setting)
#pragma config SOSCBOOST = ON           // Secondary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config POSCGAIN = GAIN_2X       // Primary Oscillator Gain Control bits (2x gain setting)
#pragma config POSCBOOST = ON           // Primary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config EJTAGBEN = NORMAL        // EJTAG Boot (Normal EJTAG functionality)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)


#include <xc.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include "ff.h"
#include "diskio.h"
#include <string.h>
#include "coder.h"
#include "mp3dec.h"


#define SYS_FREQ 200000000

#define PLAYBACK_BUFFER_SIZE 2304 * 4                  // Due to the way my circular buffer works, this *must* be a multiple of 2304

#define MP3_BUF_SIZE 1940

unsigned char MP3_READ_BUFFER[MP3_BUF_SIZE];            // Stores a chunk read from the MP3 file
unsigned char *MP3_READ_BUFFER_PTR;                     // Stores where we are in MP3_READ_BUFFER
signed short MP3_OUTPUT_BUFFER[1152 * 2];               // Stores the output from the MP3 decoder

signed short playback_buffer[PLAYBACK_BUFFER_SIZE];   // The actual data to play back
volatile int pb_readpos;                                // The read position in the playback buffer
volatile int pb_writepos;                               // The write position in the playback buffer
char active;                                            // Allow the DAC to play audio data?

HMP3Decoder mp3Decoder;                                 // The MP3 decoder itself

FIL file;                                               // file is the handle of the MP3 file we are streaming from SD card
DIR dir;                                                // Stores information about the current directory
FATFS fso;                                              // Stores information about the file system
FILINFO fileInfo;                                       // Stores information about the current file

long set_performance_mode()
{   
	unsigned int cp0;
	
    // Unlock Sequence
    asm volatile("di"); // Disable all interrupts
    SYSKEY = 0xAA996655;
    SYSKEY = 0x556699AA;  

    // PB1DIV
    // Peripheral Bus 1 cannot be turned off, so there's no need to turn it on
    PB1DIVbits.PBDIV = 1; // Peripheral Bus 1 Clock Divisor Control (PBCLK1 is SYSCLK divided by 2)

    // PB2DIV
    PB2DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB2DIVbits.PBDIV = 1; // Peripheral Bus 2 Clock Divisor Control (PBCLK2 is SYSCLK divided by 2)

    // PB3DIV
    PB3DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB3DIVbits.PBDIV = 1; // Peripheral Bus 3 Clock Divisor Control (PBCLK3 is SYSCLK divided by 2)

    // PB4DIV
    PB4DIVbits.ON = 1; // Peripheral Bus 4 Output Clock Enable (Output clock is enabled)
    while (!PB4DIVbits.PBDIVRDY); // Wait until it is ready to write to
    PB4DIVbits.PBDIV = 0; // Peripheral Bus 4 Clock Divisor Control (PBCLK4 is SYSCLK divided by 1)

    // PB5DIV
    PB5DIVbits.ON = 1; // Peripheral Bus 5 Output Clock Enable (Output clock is enabled)
    PB5DIVbits.PBDIV = 1; // Peripheral Bus 5 Clock Divisor Control (PBCLK5 is SYSCLK divided by 2)

    // PB7DIV
    PB7DIVbits.ON = 1; // Peripheral Bus 7 Output Clock Enable (Output clock is enabled)
    PB7DIVbits.PBDIV = 0; // Peripheral Bus 7 Clock Divisor Control (PBCLK7 is SYSCLK divided by 1)

    // PB8DIV
    PB8DIVbits.ON = 1; // Peripheral Bus 8 Output Clock Enable (Output clock is enabled)
    PB8DIVbits.PBDIV = 1; // Peripheral Bus 8 Clock Divisor Control (PBCLK8 is SYSCLK divided by 2)

    // PRECON - Set up prefetch
    PRECONbits.PFMSECEN = 0; // Flash SEC Interrupt Enable (Do not generate an interrupt when the PFMSEC bit is set)
    PRECONbits.PREFEN = 0b11; // Predictive Prefetch Enable (Enable predictive prefetch for any address)
    PRECONbits.PFMWS = 0b010; // PFM Access Time Defined in Terms of SYSCLK Wait States (Two wait states)

    // Set up caching
    cp0 = _mfc0(16, 0);
    cp0 &= ~0x07;
    cp0 |= 0b011; // K0 = Cacheable, non-coherent, write-back, write allocate
    _mtc0(16, 0, cp0);  

    // Lock Sequence
    SYSKEY = 0x33333333;
    
    asm volatile("ei"); // Enable all interrupts
}

void setup_ports()
{
    // Turn off all analog inputs, make everything digital
    ANSELA = 0;
    ANSELB = 0;
    ANSELC = 0;
    ANSELD = 0;
    ANSELE = 0;
    ANSELF = 0;
    ANSELG = 0;
    ANSELH = 0;
    ANSELJ = 0;
    
    // Make all ports outputs initially
    TRISA = 0;
    TRISB = 0;
    TRISC = 0;
    TRISD = 0;
    TRISE = 0;
    TRISF = 0;
    TRISG = 0;
    TRISH = 0;
    TRISJ = 0;        
    
    // Set all port values low initially
    LATA = 0;
    LATB = 0;
    LATC = 0;
    LATD = 0;
    LATE = 0;
    LATF = 0;
    LATG = 0;
    LATH = 0;
    LATJ = 0;
}

void initDisk()
{// Wait for the disk to initialise
    while(disk_initialize(0));
    // Mount the disk
    f_mount(&fso, "/", 0);
    // Change dir to the root directory
    f_chdir("/");
    // Open the directory
    f_opendir(&dir, ".");
}

int find_first_frame()
{
    int nRead;
    
    int offset;
    
    nRead = 1;
    
    // Find a valid MP3 start of frame in the input stream
    while(nRead > 0)
    {
        // Read the input file
        f_read(&file, MP3_READ_BUFFER, MP3_BUF_SIZE, &nRead);
        if(nRead == 0)
        {
            // We have reached end of file and a valid MP3 start of frame was not found.
            return -1;
        }
        else
        {
            offset = MP3FindSyncWord(MP3_READ_BUFFER, MP3_BUF_SIZE);
            if(offset < 0)
            {
                // The input buffer does not contain a start of frame. Read another frame.
                continue;
            }
            else
            {
                // We found a start of frame. offset contains location of the start of frame
                 // within input buffer.
                return offset;
            }
        }
    }    
}

static int fill_read_buffer(unsigned char *readBuf, unsigned char *readPtr, int bufSize, int bytesLeft, FIL *infile)
{
    /* This function will move bytesLeft bytes from
     * readPtr to the top of readBuf. bufSize is the
     * size of readBuf. It then reads bufSize - bytesLeft bytes
     * from infile and appends these bytes to readBuf. 
     * If readBuf is not full, then remaing bytes are
     * set to zero. The total number of bytes read from 
     * infile is returned.
     */

    int nRead;

    /* move last, small chunk from end of buffer to start, then fill with new data */
    memmove(readBuf, readPtr, bytesLeft);				
    
    f_read(&file, readBuf + bytesLeft, bufSize - bytesLeft, &nRead);

    /* zero-pad to avoid finding false sync word after last frame (from old data in readBuf) */
    if (nRead < bufSize - bytesLeft)
        memset(readBuf + bytesLeft + nRead, 0, bufSize - bytesLeft - nRead);	

    return nRead;
}

void __attribute__((vector(_SPI3_TX_VECTOR), interrupt(ipl3soft), nomips16)) SPI3TX_handler()
{    
	// Clear the SPI3 Transfer Interrupt Flag
    IFS4bits.SPI3TXIF = 0;

    SPI3BUF = playback_buffer[pb_readpos];

	// Increment the read position in the playback buffer
    if (active)
    	pb_readpos++;

	// Check if the read position has exceeded the length of the playback buffer. If so, restart it at 0.
	 if (pb_readpos >= PLAYBACK_BUFFER_SIZE) 
		 pb_readpos = 0;
}

void I2S_init(int frequency)
{    
    unsigned long int ref_freq;	// The desired output frequency
    int div, trim;				// RODIV and ROTRIM
    float calc;				// Fractional values
    unsigned short dummy; // Used only to clear the data from SPI3BUF
    
    /* 1. If using interrupts, disable the SPI interrupts in the respective IECx register. */  
	IEC4bits.SPI3TXIE = 0;
    
	/* 2. Stop and reset the SPI module by clearing the ON bit (SPIxCON<15>). */  
    SPI3CONbits.ON = 0;
    /* 3. Reset the SPI audio configuration register, SPIxCON2. */  
	SPI3CON2 = 0;
	/* 4. Reset the baud rate register, SPIxBRG. */  
    SPI3BRG = 0;
    
    /* 5. Clear the receive buffer. */  
    dummy = SPI3BUF;
    
    /* 6. Clear the ENHBUF bit (SPIxCON<16>) if using Standard Buffer mode or set the bit if using Enhanced Buffer mode */  
	SPI3CONbits.ENHBUF = 1;
	
	/* 7. If using interrupts, perform these additional steps:
		a) Clear the SPIx interrupt flags/events in the respective IFSx register. */  
    IFS4bits.SPI3TXIF = 0;
		
	/*	b) Write the SPIx interrupt priority and subpriority bits in the respective IPCx register. */  
    IPC39bits.SPI3TXIP = 3;
    IPC39bits.SPI3TXIS = 1;    
	
	/*	c) Set the SPIx interrupt enable bits in the respective IECx register. */  
	IEC4bits.SPI3TXIE = 1;

	/* 8. Clear the SPIROV bit (SPIxSTAT<6>). */  
	SPI3STATbits.SPIROV = 0;
	
	/* 9. Write the desired settings to the SPIxCON2 register. The AUDMOD<1:0> bits (SPIxCON2<1:0>) must be set to ?00? for I2S mode and the AUDEN bit (SPIxCON2<7>) must be set to ?1? to enable the audio protocol. */  
    SPI3CON2bits.AUDEN = 1;		// Enable Audio mode
    SPI3CON2bits.AUDMONO = 0; 	// Enable Stereo mode
    SPI3CON2bits.AUDMOD = 0; 	// Type of audio mode is I2S
    SPI3CON2bits.IGNROV = 1;	// Ignore receive overflows
    SPI3CON2bits.IGNTUR = 1;	// Ignore transfer underruns
    
	/* 10. Set the SPIxBRG baud rate register */
    SPI3BRG = 1; 				// Yields 64x the sample rate, for 16-bit stereo playback
	
	/* 11. Write the desired settings to the SPIxCON register:
		a) MSTEN (SPIxCON<5>) = 1. */  
    SPI3CONbits.MSTEN = 1;
	/* b) CKP (SPIxCON<6>) = 1. */  
    SPI3CONbits.CKP = 1;
	/* c) MODE<32,16> (SPIxCON<11:10>) = 0 for 16-bit audio channel data. */  
    SPI3CONbits.MODE16 = 0;
    SPI3CONbits.MODE32 = 0;

	/* Additional settings needed for using the reference clock */  
    SPI3CONbits.DISSDI = 1;		// Disable SDI pin  
    SPI3CONbits.FRMPOL = 0;		// Frame pulse is active-low  
    SPI3CONbits.CKE = 0;		// Serial output data changes on transition from idle clock state to active clock state
    SPI3CONbits.SMP = 1;		// Input data sampled at end of data output time
    SPI3CONbits.MCLKSEL = 1; 	// PBCLK is used by the Baud Rate Generator

	/* Before turning on SPI3, let's set up the Reference Clock 1 (REFO1CLK) */  

	// Set up REFO1CLK to be 256 x MIXER_FREQ and then times 2 again due to the formula
    calc = frequency * 256 * 2;
    
    // Calculate the values for RODIV and ROTRIM
	calc = (SYS_FREQ / 2 / calc); // At 44100Hz, this gives us 4.4288
    
    // Store the integer part in div (at 44100Hz, 4)
	div = (int)calc;
    // Subtract the integer part (at 44100Hz, 0.4288)
	calc -= div;
	// Multiply it by 512 to get it as a fraction of 512
    calc *=  512;
	// Store the integer part in trim
    trim = (int)calc;
	
    REFO1CON = 0;				// Reset the Reference Clock 1 configuration
    REFO1CONbits.RODIV = div;	// Set divider
    REFO1CONbits.ROSEL = 1;		// Source is PBCLK1
    REFO1TRIM = trim << 23; 	// Shift the bits 23 places to the left
    
	// Enable the output of Reference Clock 1
	REFO1CONbits.OE = 1;
    // Turn on Reference Clock 1
	REFO1CONbits.ON = 1;
    
	// Turn on the SPI3 peripheral
	SPI3CONbits.ON = 1;
}

void I2S_stop()
{
    // Stop the I2S and Reference Clock outputs
    SPI3CONbits.ON = 0;
    REFO1CONbits.ON = 0;
}

int main()
{
    int count;
    int error;
    int dec_error;
    MP3FrameInfo frameInfo;
    int bytesLeft;
    int offset;
    int nRead;
    char F_EOF;
    
    // Set performance to ultra rad
    set_performance_mode();
    
    // Moved all the ANSEL, TRIS and LAT settings to their own function
    setup_ports();
        
    // PPS for SD card
    RPB5R = 0b0110; 		// Set SDO2 = RB5
    TRISBbits.TRISB3 = 1; 	// Set SDI2 pin to input
    SDI2R = 0b1000; 		// SDI2 = RB3    
    
    // Enabling internal pull-up bits to counter potential problems
    CNPUBbits.CNPUB3 = 1;
    CNPUBbits.CNPUB5 = 1;
    
	// PPS for I2S outputs
    RPD15R = 0b1111; // D15 = REFCLKO1
    RPB10R = 0b0111; // B10 = SDO3
    RPB15R = 0b0111; // B15 = SS3
	
    // Enable multi-vectored interrupts mode
    INTCONbits.MVEC = 1;
    
    // Initialise SD card
    initDisk();
    
    // Reset SPI2BRG to give us maximum speed on SPI2 (50MHz)
    SPI2BRG = 0;
    
    // Open our MP3 file
    f_open(&file, "/beat.mp3", FA_READ);

    // Initialise the MP3 decoder
    mp3Decoder = MP3InitDecoder();
        
    if(mp3Decoder == 0)
    {
    // This means the memory allocation failed. This typically happens if there is not
    // enough heap memory. Recompile the code with additional heap memory.
    }
    
    // Not at the end of the file yet
    F_EOF = 0;    
    
    // Playback buffer read position is 0
    pb_readpos = 0;
    
    // Playback buffer write position is 1
    pb_writepos = PLAYBACK_BUFFER_SIZE - 2304;
    
    // Do not start playback yet
    active = 0;
    
    // Fill up the buffer with silence
    for (count = 0; count < PLAYBACK_BUFFER_SIZE; count++)
        playback_buffer[count] = 0; // The mid-point
    
    // Loop until we find a valid MP3 frame
    do
    {
        offset = -1;

        // Loop until we find a sync word
        while (offset < 0)
        {
            // Read a chunk from the file
            f_read(&file, MP3_READ_BUFFER, MP3_BUF_SIZE, &bytesLeft);
            MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;

            // If we've read 0 bytes, this means we've reached the end of the file or some error occurred
            if(bytesLeft == 0)
            {
                F_EOF = 1;
                break;
            }

            // Attempt to find the sync word in the MP3 file
            offset = MP3FindSyncWord(MP3_READ_BUFFER_PTR, MP3_BUF_SIZE);

            if(offset < 0)
            {
                // We did not find an offset, point MP3 read buffer pointer to the start of the array
                MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
            }
            else
            {
                // We found an offset, point MP3 read buffer to the start offset of the sync word
                MP3_READ_BUFFER_PTR += offset;
                // We can discard the data before the sync word
                bytesLeft -= offset;
                // Get out of the while() loop
                break;
            }
        }

        offset = -1;

        // Attempt to get frame information from the read buffer
        error = MP3GetNextFrameInfo(mp3Decoder, &frameInfo, MP3_READ_BUFFER_PTR);
    }
    while (error != 0);

    // Once we get to this point we've found a valid MP3 frame

    // Initialise I2S on SPI3 and also Reference Clock 1 at the desired frequency
    I2S_init(frameInfo.samprate);

    // Fill up the MP3 read buffer if necessary
    if (bytesLeft < MP3_BUF_SIZE)
    {
        // Fill the MP3 read buffer
        nRead = fill_read_buffer(MP3_READ_BUFFER, MP3_READ_BUFFER_PTR, MP3_BUF_SIZE, bytesLeft, &file);
        // If no data was read, we've reached the end of the file
        if (nRead == 0) F_EOF = 1;
        // Add numbers of bytes read to the total bytes left in the read buffer
        bytesLeft += nRead;
        // Point the read buffer pointer to the beginning of the array again
        MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
    }

    // Once we get to this point the actual decoding of the MP3 begins. 
    // From here it should just be about reading and decoding the MP3 and then putting that data into the playback buffer
    dec_error = MP3Decode(mp3Decoder, &MP3_READ_BUFFER_PTR, &bytesLeft, MP3_OUTPUT_BUFFER, 0);

    while ((dec_error == 0) && (!F_EOF))
    {                    
        // We have a decoded buffer, let's put it into the playback buffer
        for (count = 0; count < 2304; count++)
        {
            // Do not overwrite data that has not been read yet
            while (pb_writepos == pb_readpos);

            // MP3 data is 16-bit signed but we need to output 10-bit unsigned so convert it
            playback_buffer[pb_writepos] = MP3_OUTPUT_BUFFER[count];
            

            // Increment the position in the playback buffer
            pb_writepos++;

            // If the write position is out of bounds, reset it back to the beginning
            if (pb_writepos >= PLAYBACK_BUFFER_SIZE) 
            {
                pb_writepos = 0;
                // Only start playback once we have a completely full buffer
                active = 1;            
            }
        }
        
        // Do we have space left in the playback buffer?
        if (bytesLeft < MP3_BUF_SIZE)
        {
            // Fill the MP3 read buffer
            nRead = fill_read_buffer(MP3_READ_BUFFER, MP3_READ_BUFFER_PTR, MP3_BUF_SIZE, bytesLeft, &file);
            // If no data was read, we've reached the end of the file
            if (nRead == 0) F_EOF = 1;
            // Add numbers of bytes read to the total bytes left in the read buffer
            bytesLeft += nRead;
            // Point the read buffer pointer to the beginning of the array again
            MP3_READ_BUFFER_PTR = MP3_READ_BUFFER;
        }
        
        // Attempt to decode the data in the MP3 read buffer
        dec_error = MP3Decode(mp3Decoder, &MP3_READ_BUFFER_PTR, &bytesLeft, MP3_OUTPUT_BUFFER, 0);
    }
    
    // Stop the PWM outputs
   I2S_stop();
    
    // Close the file
    f_close(&file);
                
    // Release the decoder from memorry
    MP3FreeDecoder(mp3Decoder);
}