// PIC32MZ_7.c - Using the PMP to write to an LCD on the PIC32MZ
// by Aidan Mocke 2018-11-09
//
// PIC32MZ2048EFH144 Configuration Bit Settings

// DEVCFG3
// USERID = No Setting
#pragma config FMIIEN = ON              // Ethernet RMII/MII Enable (MII Enabled)
#pragma config FETHIO = ON              // Ethernet I/O Pin Select (Default Ethernet I/O)
#pragma config PGL1WAY = OFF            // Permission Group Lock One Way Configuration (Allow multiple reconfigurations)
#pragma config PMDL1WAY = OFF           // Peripheral Module Disable Configuration (Allow multiple reconfigurations)
#pragma config IOL1WAY = OFF            // Peripheral Pin Select Configuration (Allow multiple reconfigurations)
#pragma config FUSBIDIO = OFF           // USB USBID Selection (Controlled by Port Function)

// DEVCFG2
#pragma config FPLLIDIV = DIV_3         // System PLL Input Divider (3x Divider)
#pragma config FPLLRNG = RANGE_5_10_MHZ // System PLL Input Range (5-10 MHz Input)
#pragma config FPLLICLK = PLL_POSC      // System PLL Input Clock Selection (POSC is input to the System PLL)
#pragma config FPLLMULT = MUL_50        // System PLL Multiplier (PLL Multiply by 50)
#pragma config FPLLODIV = DIV_2         // System PLL Output Clock Divider (2x Divider)
#pragma config UPLLFSEL = FREQ_24MHZ    // USB PLL Input Frequency Selection (USB PLL input is 24 MHz)

// DEVCFG1
#pragma config FNOSC = SPLL             // Oscillator Selection Bits (System PLL)
#pragma config DMTINTV = WIN_127_128    // DMT Count Window Interval (Window/Interval value is 127/128 counter value)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disable SOSC)
#pragma config IESO = OFF               // Internal/External Switch Over (Disabled)
#pragma config POSCMOD = EC             // Primary Oscillator Configuration (External clock mode)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FCKSM = CSECME           // Clock Switching and Monitor Selection (Clock Switch Enabled, FSCM Enabled)
#pragma config WDTPS = PS1048576        // Watchdog Timer Postscaler (1:1048576)
#pragma config WDTSPGM = STOP           // Watchdog Timer Stop During Flash Programming (WDT stops during Flash programming)
#pragma config WINDIS = NORMAL          // Watchdog Timer Window Mode (Watchdog Timer is in non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled)
#pragma config FWDTWINSZ = WINSZ_25     // Watchdog Timer Window Size (Window size is 25%)
#pragma config DMTCNT = DMT31           // Deadman Timer Count Selection (2^31 (2147483648))
#pragma config FDMTEN = OFF             // Deadman Timer Enable (Deadman Timer is disabled)

// DEVCFG0
#pragma config DEBUG = OFF              // Background Debugger Enable (Debugger is disabled)
#pragma config JTAGEN = OFF             // JTAG Enable (JTAG Disabled)
#pragma config ICESEL = ICS_PGx1        // ICE/ICD Comm Channel Select (Communicate on PGEC1/PGED1)
#pragma config TRCEN = OFF              // Trace Enable (Trace features in the CPU are disabled)
#pragma config BOOTISA = MIPS32         // Boot ISA Selection (Boot code and Exception code is MIPS32)
#pragma config FECCCON = OFF_UNLOCKED   // Dynamic Flash ECC Configuration (ECC and Dynamic ECC are disabled (ECCCON bits are writable))
#pragma config FSLEEP = OFF             // Flash Sleep Mode (Flash is powered down when the device is in Sleep mode)
#pragma config DBGPER = PG_ALL          // Debug Mode CPU Access Permission (Allow CPU access to all permission regions)
#pragma config SMCLR = MCLR_NORM        // Soft Master Clear Enable bit (MCLR pin generates a normal system Reset)
#pragma config SOSCGAIN = GAIN_2X       // Secondary Oscillator Gain Control bits (2x gain setting)
#pragma config SOSCBOOST = ON           // Secondary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config POSCGAIN = GAIN_2X       // Primary Oscillator Gain Control bits (2x gain setting)
#pragma config POSCBOOST = ON           // Primary Oscillator Boost Kick Start Enable bit (Boost the kick start of the oscillator)
#pragma config EJTAGBEN = NORMAL        // EJTAG Boot (Normal EJTAG functionality)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)


#include <xc.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <proc/p32mz2048efh144.h>
#include "tux.h"

#define SYS_FREQ 200000000              // Running at 200MHz

#define LCD_RESET LATAbits.LATA5        // LCD's reset pin is connected to RA5

void PMP_init()
{
    asm volatile("di"); 	// Disable all interrupts             

    PMCONCLR = 0x8000; 		// Turn PMP peripheral off before setting it up
    PMAEN = 0;				// Disable all address bits
    PMAENSET = 1;			// Enable Address Bit 0 (PMA0)

    PMCONbits.WRSP = 0; 	// Set Write Strobe polarity to active low
    PMCONbits.RDSP = 0; 	// Ret Write Strobe polarity to active low
    PMCONbits.PTRDEN = 1; 	// Enable Read Strobe (RS)
    PMCONbits.PTWREN = 1; 	// Enable Write Strobe (WR)
    PMCONbits.ADRMUX = 0; 	// No MUXing please, all 16-bits on PMD<0:15>
    PMCONbits.SIDL = 0;		// Continue operation in idle mode
    PMCONbits.CSF = 00; 	// PMCS2 enabled, PMCS1 is PMA14
    PMCONbits.ALP = 0;		// Set Address Latch Polarity bit active low
    PMCONbits.DUALBUF = 0;	// Disable Dual Buffer mode (use PMDIN and PMADDR registers)
    
    PMMODEbits.IRQM = 1; 	// IRQ at the end of the Read/Write cycle
    PMMODEbits.MODE16 = 1;	// 16-bit mode
    PMMODEbits.MODE = 0b10; // Master mode 2
    PMMODEbits.WAITB = 3; 	// Maximum wait / slowest speed
    PMMODEbits.WAITM = 15; 	// Maximum wait / slowest speed
    PMMODEbits.WAITE = 3; 	// Maximum wait / slowest speed
    PMMODEbits.INCM = 00;	// Do not automatically increment address when writing

    PMCONSET = 0x8000; 		// Turn PMP peripheral on
    asm volatile("ei"); 	// Enable all interrupts again
}

void delay_us(unsigned int us)
{
    // Convert microseconds us into how many clock ticks it will take
    us *= SYS_FREQ / 1000000 / 2; // Core Timer updates every 2 ticks

    _CP0_SET_COUNT(0); // Set Core Timer count to 0

    while (us > _CP0_GET_COUNT()); // Wait until Core Timer count reaches the number we calculated earlier
}

void delay_ms(int ms)
{
    delay_us(ms * 1000);
}

static inline PMP_wait(void)
{
    while(PMMODEbits.BUSY); // Wait for the PMP to be ready
}

void LCD_write_command(unsigned short wcmd)
{
    PMP_wait();     // Wait for the PMP to be ready
    PMADDR = 0;     // PMADDR = 0 means the LCD's data/command pin will be set to 0, which means I'm sending a command
    PMP_wait();     // Wait for the PMP to be ready
    PMDIN = wcmd;   // send the data
}

void LCD_write_data(unsigned short wdata)
{
    PMP_wait();  
    PMADDR = 1;         // PMADDR = 1 means the LCD's data/command pin will be set to 1, which means I'm sending data
    PMP_wait();           // Wait for the PMP to be ready
    PMDIN = wdata;        // Send the data
    PMP_wait();           // Wait for the PMP to be ready
}

void LCD_write_command_Data(unsigned int Wcommand,unsigned int Wdata)
{
   LCD_write_command(Wcommand);
   LCD_write_data(Wdata);
}

void LCD_set_address(int PX1,int PY1,int PX2,int PY2)
{
    LCD_write_command(0x2A);
    LCD_write_data(PX1 >> 8);
    LCD_write_data(PX1 & 0xFF);
    LCD_write_data(PX2 >> 8);
    LCD_write_data(PX2 & 0xFF);
    
    LCD_write_command(0x2B);
    LCD_write_data(PY1 >> 8);
    LCD_write_data(PY1 & 0xFF);
    LCD_write_data(PY2 >> 8);
    LCD_write_data(PY2 & 0xFF);
}

void LCD_init()
{
	LCD_RESET=1;
	delay_ms(5);
	LCD_RESET=0;
	delay_ms(15);
	LCD_RESET=1;
	delay_ms(15);
	LCD_write_command(0x00E2);	// PLL multiplier, set PLL clock to 120M
	LCD_write_data(0x0023);     // N=0x36 for 6.5M, 0x23 for 10M crystal
	LCD_write_data(0x0002);
	LCD_write_data(0x0054);
	LCD_write_command(0x00E0);  // PLL enable
	LCD_write_data(0x0001);
	delay_ms(10);
	LCD_write_command(0x00E0);
	LCD_write_data(0x0003);     // Use PLL output as system clock
	delay_ms(10);
	LCD_write_command(0x0001);  // Software reset
	delay_ms(20);
	LCD_write_command(0x00E6);  // PLL setting for PCLK, depends on resolution
	LCD_write_data(0x0004);
	LCD_write_data(0x00FF);
	LCD_write_data(0x00E0);

	LCD_write_command(0x00B0);  // LCD parameters
	LCD_write_data(0x0020);     // 24 bit TFT panel
	LCD_write_data(0x0000);     // Hsync + Vsync + DE mode TFT mode
	LCD_write_data((799>>8));   // Set HDP
	LCD_write_data(799);
	LCD_write_data(479>>8);     // Set VDP
	LCD_write_data(479);
	LCD_write_data(0x0000);

	LCD_write_command(0x00B4);  // HSYNC
	LCD_write_data(0x04);       // Set HT
	LCD_write_data(0x1f);
	LCD_write_data(0x00);       // Set HPS
	LCD_write_data(0xd2);
	LCD_write_data(0x00);       // Set HPW
	LCD_write_data(0x00);       // Set HPS
	LCD_write_data(0x00);
	LCD_write_data(0x00);

	LCD_write_command(0x00B6);  // VSYNC
	LCD_write_data(0x02);       // Set VT
	LCD_write_data(0x0C);
	LCD_write_data(0x00);       // Set VPS
	LCD_write_data(0x22);
	LCD_write_data(0x00);       // Set VPW
	LCD_write_data(0x00);       // Set FPS
	LCD_write_data(0x00);


	LCD_write_command(0x00B8);
	LCD_write_data(0x000f);     // GPIO is controlled by host GPIO[3:0]=output   GPIO[0]=1  LCD ON  GPIO[0]=1  LCD OFF 
	LCD_write_data(0x0001);     // GPIO0 normal

	LCD_write_command(0x00BA);
	LCD_write_data(0x0001);     // GPIO[0] out 1 --- LCD display on/off control PIN

	LCD_write_command(0x0036);  // Rotation / Colour information
	LCD_write_data(0x000A);     // RGB=BGR

	LCD_write_command(0x003A);  // Set the current pixel format for RGB image data
	LCD_write_data(0x0050);     // 16-bits/pixel

	LCD_write_command(0x00F0);  // Pixel Data Interface Format
	LCD_write_data(0x0003);     // 16-bit(565 format) data 

	LCD_write_command(0x00BC); 
	LCD_write_data(0x0040);     // Contrast value
	LCD_write_data(0x0080);     // Brightness value
	LCD_write_data(0x0040);     // Saturation value
	LCD_write_data(0x0001);     // Post Processor Enable

	delay_ms(5);

	LCD_write_command(0x0029);  // Display on

	LCD_write_command(0x00BE);  // Set PWM for B/L
	LCD_write_data(0x0006);
	LCD_write_data(0x0080);
	LCD_write_data(0x0001);
	LCD_write_data(0x00f0);
	LCD_write_data(0x0000);
	LCD_write_data(0x0000);

	LCD_write_command(0x00D0); 
	LCD_write_data(0x000D);	
}

void set_performance_mode()
{   
	unsigned int cp0;
	
    // Unlock Sequence
    asm volatile("di"); // Disable all interrupts
    SYSKEY = 0xAA996655;
    SYSKEY = 0x556699AA;  

    // PB1DIV
    // Peripheral Bus 1 cannot be turned off, so there's no need to turn it on
    PB1DIVbits.PBDIV = 1; // Peripheral Bus 1 Clock Divisor Control (PBCLK1 is SYSCLK divided by 2)

    // PB2DIV
    PB2DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB2DIVbits.PBDIV = 1; // Peripheral Bus 2 Clock Divisor Control (PBCLK2 is SYSCLK divided by 2)

    // PB3DIV
    PB3DIVbits.ON = 1; // Peripheral Bus 2 Output Clock Enable (Output clock is enabled)
    PB3DIVbits.PBDIV = 1; // Peripheral Bus 3 Clock Divisor Control (PBCLK3 is SYSCLK divided by 2)

    // PB4DIV
    PB4DIVbits.ON = 1; // Peripheral Bus 4 Output Clock Enable (Output clock is enabled)
    while (!PB4DIVbits.PBDIVRDY); // Wait until it is ready to write to
    PB4DIVbits.PBDIV = 0; // Peripheral Bus 4 Clock Divisor Control (PBCLK4 is SYSCLK divided by 1)

    // PB5DIV
    PB5DIVbits.ON = 1; // Peripheral Bus 5 Output Clock Enable (Output clock is enabled)
    PB5DIVbits.PBDIV = 1; // Peripheral Bus 5 Clock Divisor Control (PBCLK5 is SYSCLK divided by 2)

    // PB7DIV
    PB7DIVbits.ON = 1; // Peripheral Bus 7 Output Clock Enable (Output clock is enabled)
    PB7DIVbits.PBDIV = 0; // Peripheral Bus 7 Clock Divisor Control (PBCLK7 is SYSCLK divided by 1)

    // PB8DIV
    PB8DIVbits.ON = 1; // Peripheral Bus 8 Output Clock Enable (Output clock is enabled)
    PB8DIVbits.PBDIV = 1; // Peripheral Bus 8 Clock Divisor Control (PBCLK8 is SYSCLK divided by 2)

    // PRECON - Set up prefetch
    PRECONbits.PFMSECEN = 0; // Flash SEC Interrupt Enable (Do not generate an interrupt when the PFMSEC bit is set)
    PRECONbits.PREFEN = 0b11; // Predictive Prefetch Enable (Enable predictive prefetch for any address)
    PRECONbits.PFMWS = 0b010; // PFM Access Time Defined in Terms of SYSCLK Wait States (Two wait states)

    // Set up caching
    cp0 = _mfc0(16, 0);
    cp0 &= ~0x07;
    cp0 |= 0b011; // K0 = Cacheable, non-coherent, write-back, write allocate
    _mtc0(16, 0, cp0);  

    // Lock Sequence
    SYSKEY = 0x33333333;
    asm volatile("ei"); // Enable all interrupts
}

int main()
{
    int cnt;
        
    set_performance_mode();

    // Set all port pins to digital
    ANSELA = 0;
    ANSELB = 0;
    ANSELC = 0;
    ANSELD = 0;
    ANSELE = 0;
    ANSELF = 0;
    ANSELG = 0;
    ANSELH = 0;
    ANSELJ = 0;
    
    // Set all port pins to outputs
    TRISA = 0;
    TRISB = 0;
    TRISC = 0;
    TRISD = 0;
    TRISE = 0;
    TRISF = 0;
    TRISG = 0;
    TRISH = 0;
    TRISJ = 0;
               
    // Initialise PMP
    PMP_init();
    
    // Enable the LCD by setting the Chip Select to 0
    LATGbits.LATG9 = 0;
    
    // D6 is connected to my LCD's backlight. 1 turns it on at maximum brightness, 0 turns it off and anything in between (using PWM) can control the brightness.
    LATDbits.LATD6 = 1; // Turn on BL, maximum brightness.

    // Initialise the LCD
    LCD_init();        
    
    // After LCD initialisation, re-configure PMP to use minimum wait times
    PMP_wait();           // Wait for the PMP to be free
    PMMODEbits.WAITB = 0; // Minimum wait
    PMMODEbits.WAITM = 0; // Minimum wait
    PMMODEbits.WAITE = 0; // Minimum wait

    // First, clear the screen to get rid of the mess caused by init
    LCD_set_address(0,0,799,479);       // LCD dimensions are 800 x 480 pixels, but we start at 0 so subtract 1 from each dimension
    LCD_write_command(0x2C);       		// For the SSD1963 command 0x2C enables writing of pixel data
    PMADDR = 1; 						// PMADDR = 1 means writing data, not command
    for (cnt = 0; cnt < 800 * 480; cnt++)
    {
        PMP_wait(); 					// Wait for the PMP to be free
        PMDIN = 0;                      // Draw a black pixel
    }
   
    // Now, draw the picture stored in the Tux[] array
    LCD_set_address(0,0,210-1,248-1); 	// Tux image is 210 x 248 pixels, but we start at 0 so subtract 1 from each dimension
    LCD_write_command(0x2C); 			// For the SSD1963 command 0x2C enables writing of pixel data
    PMADDR = 1; 						// PMADDR = 1 means writing data, not command
    for (cnt = 0; cnt < 210 * 248; cnt++)
    {
        PMP_wait(); 					// Wait for the PMP to be free
        PMDIN = Tux[cnt];				// Send across 16bits of data
    }

    while (1)
    {
        // Loop endlessly
    }                
}