/*------------------------------------------------------------------------/
/  MMCv3/SDv1/SDv2 (in SPI mode) control module
/-------------------------------------------------------------------------/
/
/  Copyright (C) 2010, ChaN, all right reserved.
/
/  This software is a free software and there is NO WARRANTY.
/  No restriction on use. You can use, modify and redistribute it for
/  personal, non-profit or commercial products UNDER YOUR RESPONSIBILITY.
/  Redistributions of source code must retain the above copyright notice.
/
/-------------------------------------------------------------------------*/

//Edited by A. Morrison to function on PIC32.
// Ported by Riccardo Leonardi to PIC32MX795F512L  (22/11/2011)
// Many thanks to Aiden Morrison's good work!
// changes: parametrization of SPI port number

// Modified by Bryn Thomas (11/09/2016) to use Enhanced Buffer SPI mode
// and boost read performance with 32-bit transfers

// Ported by Aidan Mocke (04/09/2018) to work on the PIC32MZ series
// Modified by Aidan Mocke (07/01/2019) to use DMA reads
// Modified by Aidan Mocke (14/01/2019) to use #defines to make config easier
// Special thanks to Bryn Thomas and Ivo Colleoni for their help with the #defines
// Configuration settings moved to mmcpic32.h, which is diskio.h renamed

#include <xc.h>
#include "mmcpic32.h"
#include "integer.h"

// Send [dat] over SPI and return the reply
static BYTE xchg_spi (BYTE dat)
{
	SPI_CHAN_FUNC(BUF) = dat;
	while (SPI_CHAN_FUNC(STATbits).SPIRBE);
	return (BYTE)SPI_CHAN_FUNC(BUF);
}

// Wait for card to be ready
static BYTE wait_ready (void)
{
	BYTE res;

	rcvr_spi();
	do
    {
		res = rcvr_spi();
    }
	while ((res != 0xFF));

	return res;
}

// Deselect the card and release SPI bus
static void deselect (void)
{
	CS_HIGH();
	rcvr_spi();
}


// Select the card and wait until it is ready
// Returns 1: Successful, 0: Error
static int select (void)	
{
	CS_LOW();
	if (wait_ready() != 0xFF) 
    {
		deselect();
		return 0;
	}
	return 1;
}

// Write sectors
// Returns 1:OK, 0:Failed 
#if _READONLY == 0
static int xmit_datablock (   
        const BYTE *buff,       // 512 byte data block to be transmitted 
        BYTE token              // Data token 
)
{
	BYTE resp;
	UINT bc = 512;

	if (wait_ready() != 0xFF) return 0;

	xmit_spi(token);		// Xmit a token 
	if (token != 0xFD) {	// Not StopTran token 
		do {						// Xmit the 512 byte data block to the MMC 
			xmit_spi(*buff++);
			xmit_spi(*buff++);
		} while (bc -= 2);
		xmit_spi(0xFF);				// CRC (Dummy) 
		xmit_spi(0xFF);
		resp = rcvr_spi();			// Receive a data response 
		if ((resp & 0x1F) != 0x05)	// If not accepted, return with error 
			return 0;
	}

	return 1;
}
#endif	// _READONLY 

static BYTE send_cmd (
	BYTE cmd,		// Command byte 
	DWORD arg		// Argument 
)
{
	BYTE n, res;


	if (cmd & 0x80) {	// ACMD<n> is the command sequence of CMD55-CMD<n> 
		cmd &= 0x7F;
		res = send_cmd(CMD55, 0);
		if (res > 1) return res;
	}

	// Select the card and wait for ready 
	deselect();
	if (!select()) return 0xFF;

	// Send command packet 
	xmit_spi(0x40 | cmd);			// Start + Command index 
	xmit_spi((BYTE)(arg >> 24));	// Argument[31..24] 
	xmit_spi((BYTE)(arg >> 16));	// Argument[23..16] 
	xmit_spi((BYTE)(arg >> 8));		// Argument[15..8] 
	xmit_spi((BYTE)arg);			// Argument[7..0] 
	n = 0x01;						// Dummy CRC + Stop 
	if (cmd == CMD0) n = 0x95;		// Valid CRC for CMD0(0) 
	if (cmd == CMD8) n = 0x87;		// Valid CRC for CMD8(0x1AA) 
	xmit_spi(n);

	// Receive command response 
	if (cmd == CMD12) rcvr_spi();	// Skip a stuff byte when stop reading 
	n = 10;							// Wait for a valid response in timeout of 10 attempts 
	do
		res = rcvr_spi();
	while ((res & 0x80) && --n);

	return res;			// Return with the response value 
}


#if _READONLY == 0
DRESULT disk_write (BYTE pdrv, const BYTE* buff, DWORD sector, UINT count)
{
	if (pdrv || !count) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;
	if (Stat & STA_PROTECT) return RES_WRPRT;

	if (!(CardType & CT_BLOCK)) sector *= 512;	// Convert to byte address if needed 

	if (count == 1) {		// Single block write 
		if ((send_cmd(CMD24, sector) == 0)	// WRITE_BLOCK 
			&& xmit_datablock(buff, 0xFE))
			count = 0;
	}
	else {				// Multiple block write 
		if (CardType & CT_SDC) send_cmd(ACMD23, count);
		if (send_cmd(CMD25, sector) == 0) {	// WRITE_MULTIPLE_BLOCK 
			do {
				if (!xmit_datablock(buff, 0xFC)) break;
				buff += 512;
			} while (--count);
			if (!xmit_datablock(0, 0xFD))	// STOP_TRAN token 
				count = 1;
		}
	}
	deselect();

	return count ? RES_ERROR : RES_OK;
}
#endif // _READONLY 

extern __inline__ unsigned int __attribute__((always_inline)) virt_to_phys(const void* p) 
{ 
 return (int)p<0?((int)p&0x1fffffffL):(unsigned int)((unsigned char*)p+0x40000000L); 
}


void SPI_DMA_init(void)
{
    DCH_RX_CHAN_FUNC(SSA) = virt_to_phys((void*)&SPI_CHAN_FUNC(BUF));           // Source address
    DCH_RX_CHAN_FUNC(ECONbits).CHSIRQ = SPI_CHAN_FUNC_PREFIXED(_,_RX_VECTOR);   // Trigger cell transfer event on SPI2 Receive IRQ
    DCH_RX_CHAN_FUNC(ECONbits).CHAIRQ = SPI_CHAN_FUNC_PREFIXED(_,_FAULT_VECTOR);// Abort on SPI Fault IRQ
    
    DCH_TX_CHAN_FUNC(DSA) = virt_to_phys((void*)&SPI_CHAN_FUNC(BUF));           // Destination address
    DCH_TX_CHAN_FUNC(ECONbits).CHSIRQ = SPI_CHAN_FUNC_PREFIXED(_,_TX_VECTOR);   // Trigger cell transfer event on SPI2 Transmit IRQ
    DCH_TX_CHAN_FUNC(ECONbits).CHAIRQ = DMA_RX_INTERRUPT_FUNC(_,_VECTOR);       // Abort on DMA Receive Complete IRQ
    
    DCH_RX_CHAN_FUNC(ECONbits).SIRQEN = 1;  // Enable cell transfer event on IRQ
    DCH_RX_CHAN_FUNC(ECONbits).AIRQEN = 1;  // Enable abort transfer IRQ
    DCH_TX_CHAN_FUNC(ECONbits).SIRQEN = 1;  // Enable cell transfer event on IRQ
    DCH_TX_CHAN_FUNC(ECONbits).AIRQEN = 1;  // Enable abort transfer IRQ
    
    DCH_RX_CHAN_FUNC(SSIZ) = 1;             // Source size permanently set to 1 byte
    DCH_RX_CHAN_FUNC(CSIZ) = 1;             // Cell (transfer) size permanently set to 1 byte
    DCH_TX_CHAN_FUNC(DSIZ) = 1;             // Destination size permanently set to 1 byte
    DCH_TX_CHAN_FUNC(CSIZ) = 1;             // Cell (transfer) size permanently set to 1 byte
        
    DMA_RX_IP = DMA_RX_INT_PRIORITY;        // Set up DMA Receive Interrupt Priority
    DMA_RX_IS = DMA_RX_INT_SUBPRIORITY;     // Set up DMA Receive Interrupt Sub-priority
}

// When the target system does not support socket power control, there is nothing to do in these functions and chk_power always returns 1.
// I've used it to set the CS pin to 0 (selected), set up the SPI peripheral and initialise some DMA-related stuff
static void power_on (void)
{
	CS_SETOUT();
    SPI_CHAN_FUNC(CONbits).ON = 0;          // Turn off SPI channel
    SPI_CHAN_FUNC(CONbits).MSTEN = 1;       // Enable Master Mode
    SPI_CHAN_FUNC(CONbits).CKP = 1;         // Clock Polarity Select is high (Idle = high, active = low)
    SPI_CHAN_FUNC(CONbits).CKE = 0;         // Clock Edge Select is low, transfer out SDO on idle->active edge
    SPI_CHAN_FUNC(CONbits).SMP = 1;         // Data is sampled at the end of the data output
    SPI_CHAN_FUNC(CONbits).MODE16 = 0;      // Not 16-bit mode
    SPI_CHAN_FUNC(CONbits).MODE32 = 0;      // Not 32-bit mode
    SPI_CHAN_FUNC(CONbits).SRXISEL = 0b01;  // Set SRXISEL to generate an interrupt when the receive buffer is not empty    
    SPI_CHAN_FUNC(CONbits).STXISEL = 0b01;  // Set STXISEL to generate an interrupt when the buffer is empty
    SPI_CHAN_FUNC(CONbits).ENHBUF = 1;      // Enable Enhanced Buffers
	SPI_CHAN_FUNC(CONbits).ON = 1;          // Turn on this SPI channel
    SPI_DMA_init();                         // Initialse DMA channels
}

// Power off the SD card
static void power_off (void)
{
	select();			// Wait for card to be ready 
	deselect();         // De-select the card
	SPI_CHAN_FUNC(CONbits).ON = 0;  // Disable SPI 
	Stat |= STA_NOINIT;	// Set STA_NOINIT 
}

// SPI_DMA_wait_token sends out 0xFF and waits for the 0xFE token to come in. It will send a maximum of [num_bytes] bytes
void SPI_DMA_wait_token(unsigned char *buffer, unsigned int num_bytes)
{
    DCH_RX_CHAN_FUNC(CON) = 0;                                  // Disable DMA receive channel
    DCH_TX_CHAN_FUNC(CON) = 0;                                  // Disable DMA transmit channel

    DCH_RX_CHAN_FUNC(DSA) = virt_to_phys(buffer);               // Destination address is [buffer]]
    DCH_RX_CHAN_FUNC(CONSET) = DMA_RX_CHANNEL_PRIORITY << 16;   // Set receive channel priority
    DCH_RX_CHAN_FUNC(INTCLR) = 0xFF00FF;                        // Clear all interrupt flags and interrupt enables
    DCH_RX_CHAN_FUNC(INTSET) = 0x80000;                         // Enable the Channel Block Transfer Complete Interrupt (CHBCIE)
    DCH_RX_CHAN_FUNC(DAT) = 0xFE;                               // Stop on any byte being equal to 0xFE
    DCH_RX_CHAN_FUNC(ECONSET) = 1 << 5;                         // Enable DMA Pattern Matching
    DCH_RX_CHAN_FUNC(CONbits).CHPATLEN = 0;                     // Use an 8-bit pattern
    DCH_RX_CHAN_FUNC(DSIZ) = num_bytes;                         // Destination size is [num_bytes]
    
    DCH_TX_CHAN_FUNC(SSIZ) = num_bytes;                         // Source size is [num_bytes]
    DCH_TX_CHAN_FUNC(CONSET) = DMA_TX_CHANNEL_PRIORITY << 16;   // Set transmit channel priority
    DCH_TX_CHAN_FUNC(INTCLR) = 0xFF00FF;                        // Clear all interrupt flags and interrupt enables
    DCH_TX_CHAN_FUNC(SSA) = virt_to_phys(buffer);               // Source address is [buffer]]
    
    DMA_BUSY = 1;                                               // DMA transfer is not complete

    DMA_RX_IF_CLEAR();                                          // Clear DMA Receive Interrupt Flag
    SPI_RX_IF_CLEAR();                                          // Clear SPI Receive Interrupt Flag
    SPI_TX_IF_CLEAR();                                          // Clear SPI Transmit Interrupt Flag
    DMA_RX_IE_ON();                                             // Enable DMA Receive Interrupt
    
    DCH_RX_CHAN_FUNC(CONSET) = 1 << 7;                          // Enable DMA Receive Channel
    DCH_TX_CHAN_FUNC(CONSET) = 1 << 7;                          // Enable DMA Transmit Channel
    DCH_TX_CHAN_FUNC(ECONSET) = 1 << 7;                         // Set CFORCE on, start transmit channel working
    DMACONSET = 0x8000;                                         // Enable DMA
}


// SPI_DMA_read sends out 0xFF and reads in the returned data into [buffer], for a total of [num_bytes] byte transfers
void SPI_DMA_read(unsigned char *buffer, unsigned int num_bytes)
{
    DCH_RX_CHAN_FUNC(CON) = 0;                                  // Disable DMA receive channel
    DCH_TX_CHAN_FUNC(CON) = 0;                                  // Disable DMA transmit channel

    DCH_RX_CHAN_FUNC(DSA) = virt_to_phys(buffer);               // Destination address is [buffer]]
    DCH_RX_CHAN_FUNC(CONSET) = DMA_RX_CHANNEL_PRIORITY << 16;   // Set receive channel priority
    DCH_RX_CHAN_FUNC(INTCLR) = 0xFF00FF;                        // Clear all interrupt flags and interrupt enables
    DCH_RX_CHAN_FUNC(INTSET) = 0x80000;                         // Enable the Channel Block Transfer Complete Interrupt (CHBCIE)
    DCH_RX_CHAN_FUNC(ECONCLR) = 1 << 5;                         // Disable DMA Pattern Matching
    DCH_RX_CHAN_FUNC(DSIZ) = num_bytes;                         // Destination size is [num_bytes]

    DCH_TX_CHAN_FUNC(SSIZ) = num_bytes;                         // Source size is [num_bytes]
    DCH_TX_CHAN_FUNC(CONSET) = DMA_TX_CHANNEL_PRIORITY << 16;   // Set transmit channel priority
    DCH_TX_CHAN_FUNC(INTCLR) = 0xFF00FF;                        // Clear all interrupt flags and interrupt enables
    DCH_TX_CHAN_FUNC(SSA) = virt_to_phys(buffer);               // Source address is [buffer]]
    
    DMA_BUSY = 1;                                               // DMA transfer is not complete

    DMA_RX_IF_CLEAR();                                          // Clear DMA Receive Interrupt Flag
    SPI_RX_IF_CLEAR();                                          // Clear SPI Receive Interrupt Flag
    SPI_TX_IF_CLEAR();                                          // Clear SPI Transmit Interrupt Flag
    DMA_RX_IE_ON();                                             // Enable DMA Receive Interrupt
    
    DCH_RX_CHAN_FUNC(CONSET) = 1 << 7;                          // Enable DMA Receive Channel
    DCH_TX_CHAN_FUNC(CONSET) = 1 << 7;                          // Enable DMA Transmit Channel
    DCH_TX_CHAN_FUNC(ECONSET) = 1 << 7;                         // Set CFORCE on, start transmit channel working
    DMACONSET = 0x8000;                                         // Enable DMA
}

// DMA0_handler catches the end of the DMA interrupt, which is advised over constantly checking the flag in a loop.
void __attribute__((vector(DMA_RX_INTERRUPT_FUNC(_,_VECTOR)), interrupt(DMA_RX_IPL_FUNC(IPL,SRS)), nomips16)) DMA_RX_handler()
{
    DMA_RX_IF_CLEAR();
    SPI_TX_IF_CLEAR();
    SPI_RX_IF_CLEAR();
    DMA_RX_IE_OFF();
    DCH_RX_CHAN_FUNC(CONCLR) = 1 << 7;                     // CHEN = 1
    DCH_TX_CHAN_FUNC(CONCLR) = 1 << 7;                     // CHEN = 1

    DMA_BUSY = 0;            // DMA is done / no longer busy
}

// Receive a single 512-byte data block from the SD card
static int rcvr_datablock (	/* 1:OK, 0:Failed */
	BYTE *buff,			/* Data buffer to store received data */
	UINT btr			/* Byte count (must be multiple of 4) */
)
{
	BYTE token;


	do {							/* Wait for data packet in timeout of 100ms */
		token = rcvr_spi();
	} while (token == 0xFF);

	if(token != 0xFE) return 0;		/* If not valid data token, retutn with error */

	do {							/* Receive the data block into buffer */
		rcvr_spi_m(buff++);
		rcvr_spi_m(buff++);
		rcvr_spi_m(buff++);
		rcvr_spi_m(buff++);
	} while (btr -= 4);
	rcvr_spi();						/* Discard CRC */
	rcvr_spi();

	return 1;						/* Return with success */
}

// Enables user to set a callback function to use in their program
void set_callback(void (*func)(int stage, int args))
{
    DMA_CALLBACK = func;
}

static int rcvr_datablock_multiple(BYTE *buff, INT btr)
{
    unsigned char READ_DONE = 0;
    unsigned char READ_ERROR = 0;
    int DMA_sectors_left;
    int DMA_read_size;
    int DMA_bytes_left;
    int DMA_bytes_read;

    // Set up the DMA transfer
    DMA_sectors_left = btr >> 9;
        
    // Initialise the state machine
    READ_DONE = 0;
    DMA_BUSY = 0;
    READ_ERROR = 0;

    // Calculate how much data to read at one time
    if (btr >= 512)
        DMA_read_size = 512; // Reading 512 bytes (one sector) at a time
    else
        DMA_read_size = btr; // Reading less than 512 bytes at a time
    DMA_bytes_left = btr;
    
    // Disable the SDO pin
    SPI_CHAN_FUNC(CONbits).DISSDO = 1;
    
    // Set the SDO pin to 1 so it will always output 0b11111111 (0xFF)
    SDO_HIGH();

    while (!READ_DONE)
    {
        // Wait until the 0xFE token is received or MAX_TOKEN_WAIT_BYTES is exceeded
        SPI_DMA_wait_token(DMA_scratch_buffer, MAX_TOKEN_WAIT_BYTES);
        
        while (DMA_BUSY)
        {
            // While waiting for the 0xFE token, call the user's callback function if it is defined
            if (DMA_CALLBACK)
                DMA_CALLBACK(DMA_STAGE_WAIT_TOKEN, DCH_TX_CHAN_FUNC(SPTR));
        };
        
        if (DCH_TX_CHAN_FUNC(SPTR) > MAX_TOKEN_WAIT_BYTES) 
        {
            // 0xFE was not found in [MAX_TOKEN_WAIT_BYTES] bytes, give up
            READ_DONE = 1;
            READ_ERROR = 1;
            break;
        }
        else
        {
            // Ensure the SPI peripheral is done, may not be the case at very slow speeds
            while (SPI_CHAN_FUNC(STATbits).SPIBUSY);
            
            // How many bytes extra do we need to read?
            DMA_bytes_read = SPI_CHAN_FUNC(STATbits).RXBUFELM;
            
            // Read the extra bytes to clear out SPIBUF before continuing
            while (SPI_CHAN_FUNC(STATbits).RXBUFELM)
                *buff++ = SPI_CHAN_FUNC(BUF);

            // Read 512 bytes minus however many we just read above and then an extra 2 for CRC data
            SPI_DMA_read(buff, DMA_read_size - DMA_bytes_read + 2);
        }
        while (DMA_BUSY)
        {
            // While waiting for the 512-byte read to complete, call the user's callback function if it is defined
            if (DMA_CALLBACK)
                DMA_CALLBACK(DMA_STAGE_WAIT_READ, 512);
        };
        
        
        // Increment buffer position by 512 bytes - (number read in advance). Do not add 2, we want to overwrite the CRC.
        buff += DMA_read_size - DMA_bytes_read;

        // Sector read complete
        DMA_sectors_left--;

        // Done reading all sectors?
        if (DMA_sectors_left == 0)
            READ_DONE = 1;

        // Ensure the SPI peripheral is done, may not be the case at very slow speeds
        while (SPI_CHAN_FUNC(STATbits).SPIBUSY);
    }
    
    // Re-enable SDO/MISO
    SPI_CHAN_FUNC(CONbits).DISSDO = 0;

	if (READ_ERROR)
        return 0;
    else
        return 1;						
}

/*-----------------------------------------------------------------------*/
/* Send a data packet to MMC                                             */
/*-----------------------------------------------------------------------*/



/*-----------------------------------------------------------------------*/
/* Send a command packet to MMC                                          */
/*-----------------------------------------------------------------------*/

/*--------------------------------------------------------------------------

   Public Functions

---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------*/
/* Initialize Disk Drive                                                 */
/*-----------------------------------------------------------------------*/

DSTATUS disk_initialize (
	BYTE drv		/* Physical drive nmuber (0) */
)
{
	BYTE n, cmd, ty, ocr[4];


	power_on();							/* Force socket power on */
        FCLK_SLOW();
	for (n = 80; n; n--) rcvr_spi();	/* 80 dummy clocks */

	ty = 0;
	if (send_cmd(CMD0, 0) == 1) {			/* Enter Idle state */
		if (send_cmd(CMD8, 0x1AA) == 1) {	/* SDv2? */
			for (n = 0; n < 4; n++) ocr[n] = rcvr_spi();			/* Get trailing return value of R7 resp */
			if (ocr[2] == 0x01 && ocr[3] == 0xAA) {				/* The card can work at vdd range of 2.7-3.6V */
				while (send_cmd(ACMD41, 0x40000000));	/* Wait for leaving idle state (ACMD41 with HCS bit) */
				if (send_cmd(CMD58, 0) == 0) {			/* Check CCS bit in the OCR */
					for (n = 0; n < 4; n++) ocr[n] = rcvr_spi();
					ty = (ocr[0] & 0x40) ? CT_SD2|CT_BLOCK : CT_SD2;	/* SDv2 */
				}
			}
		} else {							/* SDv1 or MMCv3 */
			if (send_cmd(ACMD41, 0) <= 1) 	{
				ty = CT_SD1; cmd = ACMD41;	/* SDv1 */
			} else {
				ty = CT_MMC; cmd = CMD1;	/* MMCv3 */
			}
			while (send_cmd(cmd, 0));		/* Wait for leaving idle state */
			if (send_cmd(CMD16, 512) != 0)	/* Set read/write block length to 512 */
				ty = 0;
		}
	}
	CardType = ty;
	deselect();

	if (ty) {			/* Initialization succeded */
		Stat &= ~STA_NOINIT;	/* Clear STA_NOINIT */
		FCLK_FAST();
	} else {			/* Initialization failed */
		power_off();
	}

	return Stat;
}



/*-----------------------------------------------------------------------*/
/* Get Disk Status                                                       */
/*-----------------------------------------------------------------------*/

DSTATUS disk_status (
	BYTE drv		/* Physical drive nmuber (0) */
)
{
	if (drv) return STA_NOINIT;		/* Supports only single drive */
	return Stat;
}



/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/

/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/


DRESULT disk_read (BYTE pdrv, BYTE* buff, DWORD sector, UINT count)
{
	if (pdrv || !count) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;
    
	if (!(CardType & CT_BLOCK)) sector *= 512;	/* Convert to byte address if needed */

	if (count == 1) {		/* Single block read */
		if ((send_cmd(CMD17, sector) == 0)	/* READ_SINGLE_BLOCK */
			&& rcvr_datablock(buff, 512))
			count = 0;
	}
	else {				/* Multiple block read */
		if (send_cmd(CMD18, sector) == 0) {	/* READ_MULTIPLE_BLOCK */
			if (rcvr_datablock_multiple(buff, count*512)) count = 0;
			send_cmd(CMD12, 0);				/* STOP_TRANSMISSION */
		}
	}

	return count ? RES_ERROR : RES_OK;
}

/*-----------------------------------------------------------------------*/
/* Write Sector(s)                                                       */
/*-----------------------------------------------------------------------*/

#if _READONLY == 0
DRESULT w (
	BYTE drv,				/* Physical drive nmuber (0) */
	const BYTE *buff,		/* Pointer to the data to be written */
	DWORD sector,			/* Start sector number (LBA) */
	BYTE count				/* Sector count (1..255) */
)
{
	if (drv || !count) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;
	if (Stat & STA_PROTECT) return RES_WRPRT;

	if (!(CardType & CT_BLOCK)) sector *= 512;	/* Convert to byte address if needed */

	if (count == 1) {		/* Single block write */
		if ((send_cmd(CMD24, sector) == 0)	/* WRITE_BLOCK */
			&& xmit_datablock(buff, 0xFE))
			count = 0;
	}
	else {				/* Multiple block write */
		if (CardType & CT_SDC) send_cmd(ACMD23, count);
		if (send_cmd(CMD25, sector) == 0) {	/* WRITE_MULTIPLE_BLOCK */
			do {
				if (!xmit_datablock(buff, 0xFC)) break;
				buff += 512;
			} while (--count);
			if (!xmit_datablock(0, 0xFD))	/* STOP_TRAN token */
				count = 1;
		}
	}
	deselect();

	return count ? RES_ERROR : RES_OK;
}
#endif /* _READONLY */



/*-----------------------------------------------------------------------*/
/* Miscellaneous Functions                                               */
/*-----------------------------------------------------------------------*/

DRESULT disk_ioctl (
	BYTE drv,		/* Physical drive nmuber (0) */
	BYTE ctrl,		/* Control code */
	void *buff		/* Buffer to send/receive data block */
)
{
	DRESULT res;
	BYTE n, csd[16], *ptr = buff;
	DWORD csize;


	if (drv) return RES_PARERR;
	if (Stat & STA_NOINIT) return RES_NOTRDY;

	res = RES_ERROR;
	switch (ctrl) {
		case CTRL_SYNC :	/* Flush dirty buffer if present */
			if (select()) {
				deselect();
				res = RES_OK;
			}
			break;

		case GET_SECTOR_COUNT :	/* Get number of sectors on the disk (WORD) */
			if ((send_cmd(CMD9, 0) == 0) && rcvr_datablock(csd, 16)) {
				if ((csd[0] >> 6) == 1) {	/* SDv2? */
					csize = csd[9] + ((WORD)csd[8] << 8) + 1;
					*(DWORD*)buff = (DWORD)csize << 10;
				} else {					/* SDv1 or MMCv2 */
					n = (csd[5] & 15) + ((csd[10] & 128) >> 7) + ((csd[9] & 3) << 1) + 2;
					csize = (csd[8] >> 6) + ((WORD)csd[7] << 2) + ((WORD)(csd[6] & 3) << 10) + 1;
					*(DWORD*)buff = (DWORD)csize << (n - 9);
				}
				res = RES_OK;
			}
			break;

		case GET_SECTOR_SIZE :	/* Get sectors on the disk (WORD) */
			*(WORD*)buff = 512;
			res = RES_OK;
			break;

		case GET_BLOCK_SIZE :	/* Get erase block size in unit of sectors (DWORD) */
			if (CardType & CT_SD2) {	/* SDv2? */
				if (send_cmd(ACMD13, 0) == 0) {		/* Read SD status */
					rcvr_spi();
					if (rcvr_datablock(csd, 16)) {				/* Read partial block */
						for (n = 64 - 16; n; n--) rcvr_spi();	/* Purge trailing data */
						*(DWORD*)buff = 16UL << (csd[10] >> 4);
						res = RES_OK;
					}
				}
			} else {					/* SDv1 or MMCv3 */
				if ((send_cmd(CMD9, 0) == 0) && rcvr_datablock(csd, 16)) {	/* Read CSD */
					if (CardType & CT_SD1) {	/* SDv1 */
						*(DWORD*)buff = (((csd[10] & 63) << 1) + ((WORD)(csd[11] & 128) >> 7) + 1) << ((csd[13] >> 6) - 1);
					} else {					/* MMCv3 */
						*(DWORD*)buff = ((WORD)((csd[10] & 124) >> 2) + 1) * (((csd[11] & 3) << 3) + ((csd[11] & 224) >> 5) + 1);
					}
					res = RES_OK;
				}
			}
			break;

		case MMC_GET_TYPE :		/* Get card type flags (1 byte) */
			*ptr = CardType;
			res = RES_OK;
			break;

		case MMC_GET_CSD :	/* Receive CSD as a data block (16 bytes) */
			if ((send_cmd(CMD9, 0) == 0)	/* READ_CSD */
				&& rcvr_datablock(buff, 16))
				res = RES_OK;
			break;

		case MMC_GET_CID :	/* Receive CID as a data block (16 bytes) */
			if ((send_cmd(CMD10, 0) == 0)	/* READ_CID */
				&& rcvr_datablock(buff, 16))
				res = RES_OK;
			break;

		case MMC_GET_OCR :	/* Receive OCR as an R3 resp (4 bytes) */
			if (send_cmd(CMD58, 0) == 0) {	/* READ_OCR */
				for (n = 0; n < 4; n++)
					*((BYTE*)buff+n) = rcvr_spi();
				res = RES_OK;
			}
			break;

		case MMC_GET_SDSTAT :	/* Receive SD statsu as a data block (64 bytes) */
			if (send_cmd(ACMD13, 0) == 0) {	/* SD_STATUS */
				rcvr_spi();
				if (rcvr_datablock(buff, 64))
					res = RES_OK;
			}
			break;

		default:
			res = RES_PARERR;
	}

	deselect();

	return res;
}



// disk_timerproc() is supposed to be a 1ms timer, I'm just keeping it in for compatibility
void disk_timerproc (void)
{
}