/*-----------------------------------------------------------------------/
/  Low level disk interface module include file   (C)ChaN, 2014          /
/-----------------------------------------------------------------------*/

#ifndef _MMCPIC32_DEFINED
#define _MMCPIC32_DEFINED

#ifdef __cplusplus
extern "C" {
#endif


// ***************************************************
// ** CHANGE THE BELOW SETTINGS TO MATCH YOUR BOARD **
// ***************************************************
// SD card port and pin settings
#define CS_PORT H                   // Port on which CS is to be found, A - H
#define CS_PIN 12                   // Pin number of CS, 0 -15
#define SDO_PORT B                  // Port on which SDO/MOSI is to be found, A - H
#define SDO_PIN 5                   // Pin number of SDO/MOSI, 0 - 15

// SPI channel and DMA channel configuration
#define SPI_CHANNEL 2               // Channel number to use for SD card
#define DMA_RX_CHANNEL 0            // DMA channel number to use for Receiving data, 0 - 7
#define DMA_TX_CHANNEL 1            // DMA channel number to use for Transmitting data, 0 - 7
#define DMA_RX_CHANNEL_PRIORITY 3   // Priority of DMA receiving channel, 0 - 3
#define DMA_TX_CHANNEL_PRIORITY 2   // Priority of DMA transmitting channel, 0 - 3
#define DMA_RX_INT_PRIORITY 4       // Priority of DMA receive complete interrupt, 0 - 7
#define DMA_RX_INT_SUBPRIORITY 1    // Sub-priority of DMA receive complete interrupt, 0 - 3
  
#define MAX_TOKEN_WAIT_BYTES 8192               // Maximum numbers of bytes to wait for 0xFE token when starting read
static volatile unsigned char DMA_BUSY = 0;      // Cleared by interrupt handler when DMA transfer is done
__attribute__((coherent)) unsigned char DMA_scratch_buffer[MAX_TOKEN_WAIT_BYTES]; // Buffer to store token wait data in - TO BE FIXED

typedef unsigned char BYTE;      // 8-bit
typedef unsigned int UINT;       // 32-bit
typedef unsigned long int DWORD; // 64-bit
typedef BYTE	DSTATUS;         // Status of disk functions

/* Results of Disk Functions */
typedef enum {
	RES_OK = 0,		/* 0: Successful */
	RES_ERROR,		/* 1: R/W Error */
	RES_WRPRT,		/* 2: Write Protected */
	RES_NOTRDY,		/* 3: Not Ready */
	RES_PARERR		/* 4: Invalid Parameter */
} DRESULT;

/*---------------------------------------*/
/* Prototypes for disk control functions */
DSTATUS disk_initialize (BYTE pdrv);
DSTATUS disk_status (BYTE pdrv);
DRESULT disk_read (BYTE pdrv, BYTE* buff, DWORD sector, UINT count);
DRESULT disk_write (BYTE pdrv, const BYTE* buff, DWORD sector, UINT count);
DRESULT disk_ioctl (BYTE pdrv, BYTE cmd, void* buff);

/* Disk Status Bits (DSTATUS) */
#define STA_NOINIT		0x01	/* Drive not initialized */
#define STA_NODISK		0x02	/* No medium in the drive */
#define STA_PROTECT		0x04	/* Write protected */

/* Command codes for disk_ioctrl functions */
/* Generic command (Used by FatFs) */
#define CTRL_SYNC			0	/* Complete pending write process (needed at FF_FS_READONLY == 0) */
#define GET_SECTOR_COUNT	1	/* Get media size (needed at FF_USE_MKFS == 1) */
#define GET_SECTOR_SIZE		2	/* Get sector size (needed at FF_MAX_SS != FF_MIN_SS) */
#define GET_BLOCK_SIZE		3	/* Get erase block size (needed at FF_USE_MKFS == 1) */
#define CTRL_TRIM			4	/* Inform device that the data on the block of sectors is no longer used (needed at FF_USE_TRIM == 1) */

/* Generic command (Not used by FatFs) */
#define CTRL_POWER			5	/* Get/Set power status */
#define CTRL_LOCK			6	/* Lock/Unlock media removal */
#define CTRL_EJECT			7	/* Eject media */
#define CTRL_FORMAT			8	/* Create physical format on the media */

/* MMC/SDC specific ioctl command */
#define MMC_GET_TYPE		10	/* Get card type */
#define MMC_GET_CSD			11	/* Get CSD */
#define MMC_GET_CID			12	/* Get CID */
#define MMC_GET_OCR			13	/* Get OCR */
#define MMC_GET_SDSTAT		14	/* Get SD status */
#define ISDIO_READ			55	/* Read data form SD iSDIO register */
#define ISDIO_WRITE			56	/* Write data to SD iSDIO register */
#define ISDIO_MRITE			57	/* Masked write data to SD iSDIO register */

/* ATA/CF specific ioctl command */
#define ATA_GET_REV			20	/* Get F/W revision */
#define ATA_GET_MODEL		21	/* Get model name */
#define ATA_GET_SN			22	/* Get serial number */

// Definitions for MMC/SDC command
#define CMD0   (0)			// GO_IDLE_STATE 
#define CMD1   (1)			// SEND_OP_COND 
#define ACMD41 (41|0x80)	// SEND_OP_COND (SDC) 
#define CMD8   (8)			// SEND_IF_COND 
#define CMD9   (9)			// SEND_CSD 
#define CMD10  (10)			// SEND_CID 
#define CMD12  (12)			// STOP_TRANSMISSION 
#define ACMD13 (13|0x80)	// SD_STATUS (SDC) 
#define CMD16  (16)			// SET_BLOCKLEN 
#define CMD17  (17)			// READ_SINGLE_BLOCK 
#define CMD18  (18)			// READ_MULTIPLE_BLOCK 
#define CMD23  (23)			// SET_BLOCK_COUNT 
#define ACMD23 (23|0x80)	// SET_WR_BLK_ERASE_COUNT (SDC) 
#define CMD24  (24)			// WRITE_BLOCK 
#define CMD25  (25)			// WRITE_MULTIPLE_BLOCK 
#define CMD41  (41)			// SEND_OP_COND (ACMD) 
#define CMD55  (55)			// APP_CMD 
#define CMD58  (58)			// READ_OCR 

// MMC card type flags (MMC_GET_TYPE)
#define CT_MMC		0x01                // MMC ver 3 
#define CT_SD1		0x02                // SD ver 1 
#define CT_SD2		0x04                // SD ver 2 
#define CT_SDC		(CT_SD1 | CT_SD2)	// SD 
#define CT_BLOCK	0x08                // Block addressing 

// Callback function that will be called during DMA reads
void (*DMA_CALLBACK)(int stage, int args);

#define DMA_STAGE_WAIT_TOKEN 0
#define DMA_STAGE_WAIT_READ 1

// The following defines will be used to allow for simple configuration of the above settings
#define DEF_MERGER_INDIRECTOR(w,x,y,z) w ## x ## y ## z
#define DEF_MERGER(w,x,y,z) DEF_MERGER_INDIRECTOR(w,x,y,z)
#define CS_SETOUT() DEF_MERGER(TRIS,CS_PORT,CLR,) = 1 << CS_PIN
#define CS_LOW() DEF_MERGER(LAT,CS_PORT,CLR,) = 1 << CS_PIN
#define CS_HIGH() DEF_MERGER(LAT,CS_PORT,SET,) = 1 << CS_PIN
#define SDO_HIGH() DEF_MERGER(LAT,SDO_PORT,SET,) = 1 << SDO_PIN
#define SPI_CHAN_FUNC(suffix) DEF_MERGER(SPI, SPI_CHANNEL, suffix,)
#define SPI_CHAN_FUNC_PREFIXED(prefix, suffix) DEF_MERGER(prefix, SPI, SPI_CHANNEL, suffix)
#define DCH_RX_CHAN_FUNC(suffix) DEF_MERGER(DCH, DMA_RX_CHANNEL, suffix,)
#define DCH_TX_CHAN_FUNC(suffix) DEF_MERGER(DCH, DMA_TX_CHANNEL, suffix,)
#define DMA_RX_INTERRUPT_FUNC(prefix, suffix) DEF_MERGER(prefix, DMA, DMA_RX_CHANNEL, suffix)
#define DMA_RX_IPL_FUNC(prefix, suffix) DEF_MERGER(prefix, DMA_RX_INT_PRIORITY, suffix,)

// SPI receive and transmit Interrupt Flag location defines
#if (SPI_CHANNEL == 1)
    #define SPI_RX_IF_CLEAR()       IFS3CLR = _IFS3_SPI1RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS3CLR = _IFS3_SPI1TXIF_MASK
#elif (SPI_CHANNEL == 2)
    #define SPI_RX_IF_CLEAR()       IFS4CLR = _IFS4_SPI2RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS4CLR = _IFS4_SPI2TXIF_MASK
#elif (SPI_CHANNEL == 3)
    #define SPI_RX_IF_CLEAR()       IFS4CLR = _IFS4_SPI3RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS4CLR = _IFS4_SPI3TXIF_MASK
#elif (SPI_CHANNEL == 4)
    #define SPI_RX_IF_CLEAR()       IFS5CLR = _IFS5_SPI4RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS5CLR = _IFS5_SPI4TXIF_MASK
#elif (SPI_CHANNEL == 5)
    #define SPI_RX_IF_CLEAR()       IFS5CLR = _IFS5_SPI5RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS5CLR = _IFS5_SPI5TXIF_MASK
#elif (SPI_CHANNEL == 6)
    #define SPI_RX_IF_CLEAR()       IFS5CLR = _IFS5_SPI6RXIF_MASK
    #define SPI_TX_IF_CLEAR()       IFS5CLR = _IFS5_SPI6TX_MASK
#else
    #error SPI module not handled
#endif

// DMA Interrupt Priority and Interrupt Sub-priority location defines
#if (DMA_RX_CHANNEL == 0)
    #define DMA_RX_IP               IPC33bits.DMA0IP
    #define DMA_RX_IS               IPC33bits.DMA0IS
#elif (DMA_RX_CHANNEL == 1)
    #define DMA_RX_IP               IPC33bits.DMA1IP
    #define DMA_RX_IS               IPC33bits.DMA1IS
#elif (DMA_RX_CHANNEL == 2)
    #define DMA_RX_IP               IPC34bits.DMA2IP
    #define DMA_RX_IS               IPC34bits.DMA2IS
#elif (DMA_RX_CHANNEL == 3)
    #define DMA_RX_IP               IPC34bits.DMA3IP
    #define DMA_RX_IS               IPC34bits.DMA3IS
#elif (DMA_RX_CHANNEL == 4)
    #define DMA_RX_IP               IPC34bits.DMA4IP
    #define DMA_RX_IS               IPC34bits.DMA4IS
#elif (DMA_RX_CHANNEL == 5)
    #define DMA_RX_IP               IPC34bits.DMA5IP
    #define DMA_RX_IS               IPC34bits.DMA5IS
#elif (DMA_RX_CHANNEL == 6)
    #define DMA_RX_IP               IPC35bits.DMA6IP
    #define DMA_RX_IS               IPC35bits.DMA6IS
#elif (DMA_RX_CHANNEL == 7)
    #define DMA_RX_IP               IPC35bits.DMA7IP
    #define DMA_RX_IS               IPC35bits.DMA7IS
#else
    #error DMA channel not exist
#endif

// Macros to enable or disable DMA receive interrupt and clean interrupt flags

#define DMA_RX_IE_ON()          IEC4SET = DMA_RX_INTERRUPT_FUNC(_IEC4_,IE_MASK)
#define DMA_RX_IE_OFF()         IEC4CLR = DMA_RX_INTERRUPT_FUNC(_IEC4_,IE_MASK)
#define DMA_RX_IF_CLEAR()       IFS4CLR = DMA_RX_INTERRUPT_FUNC(_IFS4_,IF_MASK)
// Change the SPI port number as needed on the following 5 lines

#define	FCLK_SLOW()	SPI_CHAN_FUNC(BRG) = 128	// Set slow clock (100k-400k) 
#define	FCLK_FAST()	SPI_CHAN_FUNC(BRG) = 0		// Set fast clock (depends on the CSD) 

// Status variables for SD card
static volatile DSTATUS Stat = STA_NOINIT;	/* Disk status */
static UINT CardType;

#define xmit_spi(dat) 	xchg_spi(dat)
#define rcvr_spi()		xchg_spi(0xFF)
#define rcvr_spi_m(p)	SPI_CHAN_FUNC(BUF) = 0xFF; while (SPI_CHAN_FUNC(STATbits).SPIRBE); *(p) = (BYTE)SPI_CHAN_FUNC(BUF);

#ifdef __cplusplus
}
#endif

#endif